// scripts.js - Handles real-time calculations and tab switching for the RLF Ledger

document.addEventListener('DOMContentLoaded', () => {
    // 1. Initial Calculation on Load
    calculateAll();
    
    // 2. Add event listener to the entire table for input changes
    const memberTable = document.getElementById('memberTable');
    if (memberTable) {
        // Event delegation: listen for input and change events on the table body
        memberTable.addEventListener('input', (event) => {
            // Only recalculate if the input field is a number input
            if (event.target.tagName === 'INPUT' && (event.target.type === 'number' || event.target.type === 'text')) {
                calculateAll();
            }
        });

        // Event listener for the payment method dropdowns
        memberTable.addEventListener('change', (event) => {
            if (event.target.classList.contains('payment-method-select')) {
                calculateAll();
            }
        });
    }

    // 3. Add event listener to summary expense fields
    const summaryFields = document.querySelectorAll('#summary input[type="number"]');
    summaryFields.forEach(input => {
        input.addEventListener('input', calculateSummary);
    });

    // 4. Force calculation on initial load to set all grand totals correctly,
    // especially if B/F balances were loaded.
    const groupSelect = document.querySelector('select[name="group_id"]');
    if (groupSelect && groupSelect.value) {
        calculateAll(); 
    }
});

// --- Core Calculation Functions ---

/**
 * Parses input value to a float, defaulting to 0.00 if invalid or empty.
 */
function parseValue(value) {
    const val = parseFloat(value);
    return isNaN(val) ? 0.00 : val;
}

/**
 * Calculates a single member's row data (C/F balances, Total Repaid).
 * @param {HTMLElement} rowElement - The table row element.
 * @returns {object} The row data including totals and memberCashInflow.
 */
function calculateMember(rowElement) {
    // Safely get the row element
    const row = rowElement.closest('tr') || rowElement;

    // Get input and output elements by class name within the row
    const bfSavings = parseValue(row.querySelector('.bf-savings')?.value);
    const bfLoan = parseValue(row.querySelector('.bf-loan')?.value);
    
    const principal = parseValue(row.querySelector('.principal')?.value);
    
    // --- LOAN INTEREST CALCULATION (Uses default 5.00 if not changed) ---
    const loanInterestRate = parseValue(row.querySelector('.loan-interest-rate')?.value);
    // Loan Interest Amount = B/F Loan * (Rate / 100)
    const loanInterestAmount = bfLoan * (loanInterestRate / 100); 
    
    const shares = parseValue(row.querySelector('.shares')?.value);
    const fines = parseValue(row.querySelector('.fines')?.value);

    // NEW FIELDS COLLECTION
    const paymentMethod = row.querySelector('.payment-method-select')?.value;
    const paidAmount = parseValue(row.querySelector('.paid-amount')?.value);
    
    const withdrawal = parseValue(row.querySelector('.withdrawal')?.value);
    const newLoan = parseValue(row.querySelector('.new-loan')?.value);
    const newAdvance = parseValue(row.querySelector('.new-advance')?.value);
    
    // Total Repaid now includes the calculated interest amount
    const totalRepaid = principal + loanInterestAmount; 
    
    // C/F Calculations
    const cfSavings = bfSavings + shares - withdrawal;
    const cfLoan = bfLoan - principal + newLoan + newAdvance;
    
    // Update the Read-Only/Span fields in the row
    if (row.querySelector('.total-repaid')) row.querySelector('.total-repaid').value = totalRepaid.toFixed(2);
    if (row.querySelector('.cf-savings')) row.querySelector('.cf-savings').textContent = cfSavings.toFixed(2);
    if (row.querySelector('.cf-loan')) row.querySelector('.cf-loan').textContent = cfLoan.toFixed(2);
    
    // IMPORTANT: Update the hidden input field which is submitted
    if (row.querySelector('.loan-interest')) row.querySelector('.loan-interest').value = loanInterestAmount.toFixed(2);

    // Determine actual Cash Inflow for this member
    let memberCashInflow = 0.00;
    if (paymentMethod === 'Cash Pay') {
        memberCashInflow = paidAmount;
    }

    return { 
        totalRepaid, 
        shares, fines, 
        withdrawal, newLoan, newAdvance,
        memberCashInflow: memberCashInflow 
    };
}


/**
 * Calculates all row totals and grand totals.
 */
function calculateAll() {
    const tableBody = document.querySelector('#memberTable tbody');
    if (!tableBody) return;

    // Grand Totals initialization
    let totalRepaid = 0;
    let totalShares = 0;
    let totalFines = 0;
    let totalWithdrawal = 0;
    let totalNewLoan = 0;
    let totalNewAdvance = 0;
    let groupTotalCashInflow = 0; 

    const rows = tableBody.querySelectorAll('tr');

    rows.forEach(row => {
        const rowData = calculateMember(row);

        // Accumulate Grand Totals
        totalRepaid += rowData.totalRepaid; 
        totalShares += rowData.shares;
        totalFines += rowData.fines;
        totalWithdrawal += rowData.withdrawal;
        totalNewLoan += rowData.newLoan;
        totalNewAdvance += rowData.newAdvance;
        
        groupTotalCashInflow += rowData.memberCashInflow; 
    });

    // Update Summary Section Totals using the accumulated group totals
    updateSummarySection(
        totalRepaid, 
        totalShares, 
        totalFines, 
        totalWithdrawal, 
        totalNewLoan, 
        totalNewAdvance,
        groupTotalCashInflow 
    );
}

/**
 * Calculates and updates the Group Summary fields (Page 2) based on member totals.
 */
function calculateSummary() {
    // 1. Get Values
    const totalInflow = parseValue(document.getElementById('summaryTotalPaid')?.textContent); 
    
    const prvBanking = parseValue(document.querySelector('[name="prvBanking"]')?.value);
    const expenseMeals = parseValue(document.querySelector('[name="expenseMeals"]')?.value);
    const expenseTransport = parseValue(document.querySelector('[name="expenseTransport"]')?.value);
    const expenseStationary = parseValue(document.querySelector('[name="expenseStationary"]')?.value);
    const expenseOthers = parseValue(document.querySelector('[name="expenseOthers"]')?.value);

    // 2. Calculations
    const totalExpenses = expenseMeals + expenseTransport + expenseStationary + expenseOthers;

    // Final Reconciliation (Cash Balance = Prev Banking + Total Inflow - Total Expenses)
    const grandTotal = prvBanking + totalInflow - totalExpenses;

    // 3. Update Display fields
    if (document.getElementById('totalExpenses')) document.getElementById('totalExpenses').textContent = totalExpenses.toFixed(2);
    if (document.getElementById('grandTotal')) document.getElementById('grandTotal').textContent = grandTotal.toFixed(2);
    // Crucial: Update the hidden field that submits the grand total
    if (document.getElementById('grandTotalSubmitted')) document.getElementById('grandTotalSubmitted').value = grandTotal.toFixed(2); 
}

/**
 * Updates the totals displayed in the Group Summary tab (Page 2).
 */
function updateSummarySection(totalRepaid, totalShares, totalFines, totalWithdrawal, totalNewLoan, totalNewAdvance, totalCashInflow) { 
    
    // 1. Update Inflow Totals (Page 2 elements)
    if (document.getElementById('summaryTotalPaid')) document.getElementById('summaryTotalPaid').textContent = totalCashInflow.toFixed(2); // Use the cash flow figure
    if (document.getElementById('summaryTotalRepaid')) document.getElementById('summaryTotalRepaid').textContent = (totalRepaid).toFixed(2);
    if (document.getElementById('summaryTotalSavings')) document.getElementById('summaryTotalSavings').textContent = totalShares.toFixed(2);
    if (document.getElementById('summaryTotalFines')) document.getElementById('summaryTotalFines').textContent = totalFines.toFixed(2);

    // 2. Update Outflow Totals (Page 2 elements)
    if (document.getElementById('totalWithdrawal')) document.getElementById('totalWithdrawal').textContent = totalWithdrawal.toFixed(2);
    if (document.getElementById('totalNewLoan')) document.getElementById('totalNewLoan').textContent = totalNewLoan.toFixed(2);
    if (document.getElementById('totalNewAdvance')) document.getElementById('totalNewAdvance').textContent = totalNewAdvance.toFixed(2);
    
    // 3. Calculate Final Cash Reconciliation (which reads from the updated spans)
    calculateSummary();
}


// --- Tab Switching Function ---
function showTab(tabId) {
    // Hide all tab contents
    document.querySelectorAll('.tab-content').forEach(content => {
        content.classList.remove('active');
    });
    // Deactivate all buttons
    document.querySelectorAll('.tab-button').forEach(button => {
        button.classList.remove('active');
    });

    // Show the selected tab content and activate the button
    document.getElementById(tabId).classList.add('active');
    document.querySelector(`.tab-button[onclick="showTab('${tabId}')"]`).classList.add('active');
    
    // Recalculate summary every time the Summary tab is viewed to ensure fresh totals
    if (tabId === 'summary') {
        calculateAll(); // Forces calculation of all group totals from Page 1 data
        calculateSummary(); // Forces calculation of final cash balance
    }
}