<?php
// ledger_controller.php
require_once 'db_model.php';
require_once 'config.php';

// --- Global Data Structures ---
global $conn;

$groups_list = get_all_groups($conn);
$selected_group_id = $_GET['group_id'] ?? ($_POST['group_id'] ?? null); 
$members_data = []; 
$report_status = ''; 

// --- Helper: Get Group Name for Display ---
function get_group_name_by_id($groups, $id) {
    foreach($groups as $group) {
        if ($group['group_id'] == $id) return $group['group_name'];
    }
    return '';
}

// --- 1. Load Initial Member Data (B/F Balances) ---
function load_ledger_data() {
    global $conn, $members_data, $selected_group_id;
    
    if (!$selected_group_id) {
        return; 
    }
    
    $members = get_all_active_members_by_group($conn, $selected_group_id);
    
    foreach ($members as $member) {
        $balances = get_last_report_balances($conn, $member['member_no']);
        
        $members_data[] = [
            'member_no' => $member['member_no'], 
            'name_details' => $member['name_details'],
            'bf_savings' => $balances['cf_savings'], 
            'bf_loan' => $balances['cf_loan']
        ];
    }
}

// --- 2. Process Form Submission ---
function handle_submission() {
    global $conn, $report_status, $selected_group_id, $groups_list; 
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') return;
    
    // Check if the submission is for the ledger form
    if (!isset($_POST['save_transactions'])) return;

    // Report Date and Officer Name are now available directly from the POST data (from index.php page 1)
    $report_date = $_POST['reportDate'] ?? date('Y-m-d');
    $officer_name = $_POST['officerName'] ?? 'N/A'; // Capture officer name
    $group_name = get_group_name_by_id($groups_list, $selected_group_id);
    $total_cash_inflow = 0.00; // Initialize CASH INFLOW tracker (NEW)
    
    // Ensure group is selected before starting transaction
    if (!$selected_group_id) {
        $report_status = '<div style="color: red; font-weight: bold; background: #ffebee; padding: 10px;">ERROR: Please select a group before submitting.</div>';
        return;
    }
    
    // NOTE: For simplicity, we assume the user intends to OVERWRITE member transactions for the given date.
    // A robust system would delete existing member transactions before inserting new ones, 
    // but we skip that step for now as it's complex and not explicitly requested.

    $conn->begin_transaction();
    $transaction_successful = true;

    // A. Loop through all submitted member transactions
    if (isset($_POST['member_no'])) {
        foreach ($_POST['member_no'] as $key => $member_no) { 
            // Retrieve and sanitize data
            $member_no = intval($member_no); 
            $bf_savings = floatval($_POST['bf_savings'][$key]);
            $bf_loan = floatval($_POST['bf_loan'][$key]);
            $principal = floatval($_POST['principal'][$key]);
            $loan_interest = floatval($_POST['loan_interest'][$key]); // This is now the CALCULATED AMOUNT
            $shares = floatval($_POST['shares'][$key]);
            $fines = floatval($_POST['fines'][$key]);
            
            // --- NEW FIELDS COLLECTION ---
            $payment_method = $_POST['payment_method'][$key] ?? 'C/F Due';
            $paid_amount = floatval($_POST['paid_amount'][$key] ?? 0);
            // --- END NEW FIELDS COLLECTION ---
            
            $withdrawal = floatval($_POST['withdrawal'][$key]);
            $new_loan = floatval($_POST['new_loan'][$key]);
            $new_advance = floatval($_POST['new_advance'][$key]);
            
            // Server-side C/F Calculations (maintained)
            $cf_savings = $bf_savings + $shares - $withdrawal;
            $cf_loan = $bf_loan - $principal + $new_loan + $new_advance; 

            // CRITICAL UPDATE: Calculate Cash Inflow (only 'Cash Pay' contributes)
            if ($payment_method == 'Cash Pay') {
                $total_cash_inflow += $paid_amount;
            }

            // Prepare data for save_member_transaction 
            $member_data = [
                'member_no' => $member_no, 
                'report_date' => $report_date,
                'bf_savings' => $bf_savings,
                'bf_loan' => $bf_loan,
                'principal' => $principal,
                'loan_interest' => $loan_interest,
                'shares' => $shares,
                'fines' => $fines,
                'payment_method' => $payment_method, 
                'paid_amount' => $paid_amount,       
                'withdrawal' => $withdrawal,
                'new_loan' => $new_loan,
                'new_advance' => $new_advance,
                'cf_savings' => $cf_savings,
                'cf_loan' => $cf_loan,
            ];

            if (!save_member_transaction($conn, $member_data)) {
                $transaction_successful = false;
                break;
            }
        }
    }
    
    // B. Save/Update Group Summary Data 
    if ($transaction_successful) {
        
        // --- Calculate the server-side Grand Total ---
        $prvBanking = floatval($_POST['prvBanking'] ?? 0);
        $expenseMeals = floatval($_POST['expenseMeals'] ?? 0);
        $expenseTransport = floatval($_POST['expenseTransport'] ?? 0);
        $expenseStationary = floatval($_POST['expenseStationary'] ?? 0);
        $expenseOthers = floatval($_POST['expenseOthers'] ?? 0);
        $total_expenses = $expenseMeals + $expenseTransport + $expenseStationary + $expenseOthers;
        
        // Grand Total: (Total Cash Inflow - Total Expenses) + Previous Banking
        $final_grand_total = ($total_cash_inflow - $total_expenses) + $prvBanking;
        
        $summary_data = [
            'group_id' => $selected_group_id, // CRITICAL FIX: Add the group ID
            'report_date' => $report_date,
            'groupName' => $group_name, 
            'officerName' => $officer_name, 
            'prvBanking' => $prvBanking,
            'totalInflow' => $total_cash_inflow, 
            'expenseMeals' => $expenseMeals,
            'expenseTransport' => $expenseTransport,
            'expenseStationary' => $expenseStationary,
            'expenseOthers' => $expenseOthers,
            'nextDate' => $_POST['nextDate'] ?? null,
            'nextTime' => $_POST['nextTime'] ?? null,
            'nextVenue' => $_POST['nextVenue'] ?? 'N/A',
            'nextStage' => $_POST['nextStage'] ?? 'N/A',
            'grandTotal' => $final_grand_total 
        ];

        // NEW LOGIC: Check for existence and update instead of insert to avoid "Duplicate entry"
        $existing_summary_id = get_existing_summary_id($conn, $selected_group_id, $report_date);

        if ($existing_summary_id) {
             // Update the existing record
            if (!update_group_summary($conn, $summary_data, $existing_summary_id)) {
                $transaction_successful = false;
            }
        } else {
             // Insert a new record
            if (!save_group_summary($conn, $summary_data)) {
                $transaction_successful = false;
            }
        }
    }
    
    // C. Finalize Transaction 
    if ($transaction_successful) {
        $conn->commit();
        $report_status = '<div style="color: green; font-weight: bold; background: #e8f5e9; padding: 10px;">SUCCESS: Monthly report saved successfully!</div>';
        // Redirect to clear POST data
        header("Location: index.php?group_id=" . $selected_group_id . "&status=success"); 
        exit;
    } else {
        $conn->rollback();
        // Provide the specific database error if possible
        $db_error = $conn->error ? " DB Error: " . $conn->error : ""; 
        $report_status = '<div style="color: red; font-weight: bold; background: #ffebee; padding: 10px;">ERROR: Failed to save the report. Transaction rolled back.' . $db_error . '</div>';
    }
}

// Execute logic
handle_submission();
load_ledger_data();

// Close the connection before the HTML output
close_db_connection($conn);
?>