<?php
// db_model.php
require_once 'config.php';

// --- Group CRUD Operations ---

function get_all_groups($conn) {
    $sql = "SELECT group_id, group_name FROM groups ORDER BY group_name ASC";
    $result = $conn->query($sql);
    $groups = [];
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $groups[] = $row;
        }
    }
    return $groups;
}

function add_new_group($conn, $group_name) {
    $sql = "INSERT INTO groups (group_name, creation_date) VALUES (?, CURDATE())";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("s", $group_name);
        return $stmt->execute();
    }
    return false;
}

function update_group($conn, $group_id, $group_name) {
    $sql = "UPDATE groups SET group_name = ? WHERE group_id = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("si", $group_name, $group_id);
        return $stmt->execute();
    }
    return false;
}

function delete_group($conn, $group_id) {
    $sql = "DELETE FROM groups WHERE group_id = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $group_id);
        return $stmt->execute();
    }
    return false;
}

// --- Member CRUD Operations ---

function get_all_members($conn) {
    $sql = "SELECT m.*, g.group_name 
            FROM members m 
            JOIN groups g ON m.group_id = g.group_id 
            ORDER BY m.member_no ASC";
    $result = $conn->query($sql);
    $members = [];
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $members[] = $row;
        }
    }
    return $members;
}

function get_all_active_members_by_group($conn, $group_id) {
    $sql = "SELECT member_no, name_details, national_id, phone_number 
            FROM members 
            WHERE group_id = ? AND is_active = 1
            ORDER BY member_no ASC";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $group_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $members = [];
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                $members[] = $row;
            }
        }
        $stmt->close();
        return $members;
    }
    return [];
}

function add_new_member($conn, $group_id, $member_no, $name_details, $national_id = null, $phone_number = null) {
    $sql = "INSERT INTO members (group_id, member_no, name_details, national_id, phone_number, is_active) 
            VALUES (?, ?, ?, ?, ?, 1)";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("iissd", $group_id, $member_no, $name_details, $national_id, $phone_number);
        return $stmt->execute();
    }
    return false;
}

function update_member($conn, $member_no, $group_id, $name_details, $national_id, $phone_number, $is_active) {
    $sql = "UPDATE members SET group_id = ?, name_details = ?, national_id = ?, phone_number = ?, is_active = ? WHERE member_no = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("issdii", $group_id, $name_details, $national_id, $phone_number, $is_active, $member_no);
        return $stmt->execute();
    }
    return false;
}

function delete_member($conn, $member_no) {
    $sql = "DELETE FROM members WHERE member_no = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $member_no);
        return $stmt->execute();
    }
    return false;
}

// --- Ledger/Transaction Operations ---

function get_last_report_balances($conn, $member_no) {
    $sql = "SELECT cf_savings, cf_loan 
            FROM monthly_transactions 
            WHERE member_no = ? 
            ORDER BY report_date DESC 
            LIMIT 1";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $member_no);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($row = $result->fetch_assoc()) {
            $stmt->close();
            return $row;
        }
    }
    // Return default (zero) balances if no previous records exist
    return ['cf_savings' => 0.00, 'cf_loan' => 0.00]; 
}

function save_member_transaction($conn, $data) {
    $sql = "INSERT INTO monthly_transactions (
        member_no, report_date, bf_savings, bf_loan, 
        principal_paid, loan_interest, shares_this_month, fines_charges, 
        payment_method, paid_amount,            
        savings_withdrawal, new_loan, new_advance, 
        cf_savings, cf_loan
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"; 

    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("isddddddsdddddd", 
            $data['member_no'], $data['report_date'], $data['bf_savings'], $data['bf_loan'], 
            $data['principal'], $data['loan_interest'], $data['shares'], $data['fines'], 
            $data['payment_method'], $data['paid_amount'], 
            $data['withdrawal'], $data['new_loan'], $data['new_advance'], 
            $data['cf_savings'], $data['cf_loan']
        );
        return $stmt->execute();
    }
    return false;
}

// --- NEW FUNCTION: Check for existing group summary
function get_existing_summary_id($conn, $group_id, $report_date) {
    $sql = "SELECT summary_id FROM group_summary WHERE group_id = ? AND report_date = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("is", $group_id, $report_date);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($row = $result->fetch_assoc()) {
            $stmt->close();
            return $row['summary_id'];
        }
    }
    return null;
}

// --- UPDATE FUNCTION: Update an existing group summary
function update_group_summary($conn, $data, $summary_id) {
    $sql = "UPDATE group_summary SET 
        group_id = ?, report_date = ?, group_name = ?, officer_name = ?, prv_banking = ?, 
        total_inflow = ?, expense_meals = ?, expense_transport = ?, expense_stationary = ?, 
        expense_others = ?, next_date = ?, next_time = ?, next_venue = ?, next_stage = ?, 
        final_grand_total = ?
        WHERE summary_id = ?";
    
    if ($stmt = $conn->prepare($sql)) {
        // i sssddddddssssd i (16 total: 1i, 3s, 7d, 4s, 1d, 1i)
        $stmt->bind_param("isssddddddssssdi", 
            $data['group_id'], $data['report_date'], $data['groupName'], $data['officerName'], 
            $data['prvBanking'], $data['totalInflow'], $data['expenseMeals'], 
            $data['expenseTransport'], $data['expenseStationary'], $data['expenseOthers'], 
            $data['nextDate'], $data['nextTime'], $data['nextVenue'], $data['nextStage'], 
            $data['grandTotal'], $summary_id
        );
        return $stmt->execute();
    }
    return false;
}

// --- INSERT FUNCTION: Save a NEW group summary
function save_group_summary($conn, $data) {
    $sql = "INSERT INTO group_summary (
        group_id, report_date, group_name, officer_name, prv_banking, 
        total_inflow, 
        expense_meals, expense_transport, expense_stationary, expense_others, 
        next_date, next_time, next_venue, next_stage, final_grand_total
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"; 
    
    if ($stmt = $conn->prepare($sql)) {
        // i sssddddddssssd (15 total: 1i, 3s, 7d, 4s, 1d)
        $stmt->bind_param("isssddddddssssd", 
            $data['group_id'],
            $data['report_date'], $data['groupName'], $data['officerName'], $data['prvBanking'], 
            $data['totalInflow'],
            $data['expenseMeals'], $data['expenseTransport'], $data['expenseStationary'], $data['expenseOthers'], 
            $data['nextDate'], $data['nextTime'], $data['nextVenue'], $data['nextStage'], $data['grandTotal']
        );
        return $stmt->execute();
    }
    return false;
}


function get_group_summaries_by_filters($conn, $group_id = null, $date_from = null, $date_to = null) {
    $sql = "SELECT * FROM group_summary gs WHERE 1=1 ";
    $params = [];
    $types = "";

    if ($group_id !== null) {
        $sql .= "AND gs.group_id = ? ";
        $params[] = $group_id;
        $types .= "i";
    }

    if ($date_from !== null) {
        $sql .= "AND gs.report_date >= ? ";
        $params[] = $date_from;
        $types .= "s";
    }

    if ($date_to !== null) {
        $sql .= "AND gs.report_date <= ? ";
        $params[] = $date_to;
        $types .= "s";
    }
    
    // Default filter for better performance if no dates are provided
    if ($date_from === null && $date_to === null) {
        $sql .= "AND gs.report_date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)";
    }
    
    $sql .= " ORDER BY gs.report_date DESC";
    
    $stmt = $conn->prepare($sql);
    if ($stmt) {
        if (!empty($params)) {
            $stmt->bind_param($types, ...$params);
        }
        $stmt->execute();
        $result = $stmt->get_result();
        $summaries = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $summaries[] = $row;
            }
        }
        $stmt->close();
        return $summaries;
    }
    return [];
}

function get_detailed_member_transactions($conn, $report_date, $group_id) {
    // This query fetches transactions AND the associated member's details
    $sql = "SELECT mt.*, m.member_no, m.name_details, m.national_id, m.phone_number 
            FROM monthly_transactions mt
            JOIN members m ON mt.member_no = m.member_no
            WHERE mt.report_date = ? 
            AND m.group_id = ?
            ORDER BY m.member_no ASC";
            
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("si", $report_date, $group_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $transactions = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $transactions[] = $row;
            }
        }
        $stmt->close();
        return $transactions;
    }
    return [];
}
?>