<?php
// admin.php - Handles all Group and Member CRUD operations
require_once 'db_model.php'; // Includes config.php

global $conn;
$message = '';
$groups_list = get_all_groups($conn); 
$all_members = get_all_members($conn); 

// --- 1. Handle Group Actions ---
if (isset($_POST['add_group'])) {
    $group_name = trim($_POST['new_group_name']);
    if (!empty($group_name)) {
        if (add_new_group($conn, $group_name)) {
            $message = '<p class="success">Group "' . htmlspecialchars($group_name) . '" created successfully!</p>';
        } else {
            $message = '<p class="error">Error adding group. Check database connection or permissions.</p>';
        }
    }
} elseif (isset($_POST['update_group'])) {
    $group_id = intval($_POST['group_id']);
    $group_name = trim($_POST['group_name']);
    if (update_group($conn, $group_id, $group_name)) {
        $message = '<p class="success">Group ID ' . $group_id . ' updated successfully!</p>';
    } else {
        $message = '<p class="error">Error updating group.</p>';
    }
} elseif (isset($_GET['delete_group_id'])) {
    $group_id = intval($_GET['delete_group_id']);
    if (delete_group($conn, $group_id)) {
        $message = '<p class="success">Group ID ' . $group_id . ' deleted successfully!</p>';
    } else {
        $message = '<p class="error">Error deleting group. Ensure no members are assigned to it.</p>';
    }
}

// --- 2. Handle Member Actions (Uses member_no) ---
if (isset($_POST['add_member'])) {
    $group_id = intval($_POST['group_id']);
    $name_details = trim($_POST['member_name']);
    $national_id = trim($_POST['national_id']);
    $phone_number = trim($_POST['phone_number']);
    
    if (add_new_member($conn, $group_id, $name_details, $national_id, $phone_number)) {
        $message = '<p class="success">Member "' . htmlspecialchars($name_details) . '" created successfully with M/NO: ' . $conn->insert_id . '</p>';
    } else {
         $message = '<p class="error">Error adding member. Check if National ID is unique or database error.</p>';
    }
} elseif (isset($_POST['update_member'])) {
    $data = $_POST;
    $member_no = intval($data['member_no']); 
    $national_id = trim($data['national_id']); 
    $phone_number = trim($data['phone_number']);
    $is_active = isset($data['is_active']) ? 1 : 0;
    
    if (update_member($conn, $member_no, $data['group_id'], $data['member_name'], $national_id, $phone_number, $is_active)) {
         $message = '<p class="success">Member M/NO ' . $member_no . ' updated successfully!</p>';
    } else {
        $message = '<p class="error">Error updating member.</p>';
    }
} elseif (isset($_GET['delete_member_no'])) {
    $member_no = intval($_GET['delete_member_no']);
    if (delete_member($conn, $member_no)) {
        $message = '<p class="success">Member M/NO ' . $member_no . ' deleted successfully!</p>';
    } else {
        $message = '<p class="error">Error deleting member. Check for existing transactions.</p>';
    }
}

// Reload data after any operation to reflect changes
if (!empty($_POST) || isset($_GET['delete_group_id']) || isset($_GET['delete_member_no'])) {
    $groups_list = get_all_groups($conn);
    $all_members = get_all_members($conn);
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Redirect to clear POST data and show status
        $status = isset($message) ? 'success' : 'error';
        header("Location: admin.php?status=" . $status); 
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>SEEPO RLF - Admin Management (CRUD)</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <header>
        <h1>Admin Management (Full CRUD)</h1>
    </header>

    <div class="container">
        <?php echo $message; ?>
        
        <p style="text-align: center;">
            <a href="index.php" class="header-link">← Back to Ledger Entry</a> 
            <a href="reports.php" class="header-link" style="margin-left: 20px;">View Reports & History</a>
        </p>

        <div class="form-box">
            <details>
                <summary>
                    <h3>👥 Group Management (Create, Read, Update, Delete)</h3>
                </summary>
                <div class="details-content-wrapper">
                    <h4>➕ Add New Group</h4>
                    <form method="POST" action="admin.php">
                        <input type="text" name="new_group_name" placeholder="New Group Name" required>
                        <button type="submit" name="add_group">Add Group</button>
                    </form>
                    
                    <h4 style="margin-top: 20px;">Existing Groups</h4>
                    <table class="group-list">
                        <thead><tr><th>ID</th><th>Name</th><th>Actions</th></tr></thead>
                        <tbody>
                            <?php foreach ($groups_list as $group): ?>
                                <tr>
                                    <td><?php echo $group['group_id']; ?></td>
                                    <td>
                                        <form method="POST" action="admin.php" style="display: inline;">
                                            <input type="hidden" name="group_id" value="<?php echo $group['group_id']; ?>">
                                            <input type="text" name="group_name" value="<?php echo htmlspecialchars($group['group_name']); ?>" style="width: 250px; margin: 0;">
                                            <button type="submit" name="update_group" class="edit-btn small-btn">Update</button>
                                        </form>
                                    </td>
                                    <td>
                                        <a href="admin.php?delete_group_id=<?php echo $group['group_id']; ?>" class="action-link delete-btn small-btn" 
                                           onclick="return confirm('WARNING: Deleting this group may cause errors if members or transactions are linked. ARE YOU SURE?');">Delete</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </details>
        </div>

        <div class="form-box">
            <details>
                <summary>
                    <h3>👤 Member Management (Create, Read, Update, Delete)</h3>
                </summary>
                <div class="details-content-wrapper">
                    <h4>➕ Add New Member</h4>
                    <form method="POST" action="admin.php" style="display: flex; flex-wrap: wrap; gap: 10px; align-items: flex-end;">
                        <label>Group:
                            <select name="group_id" required>
                                <option value="">-- Assign Group --</option>
                                <?php foreach ($groups_list as $group): ?>
                                    <option value="<?php echo $group['group_id']; ?>"><?php echo htmlspecialchars($group['group_name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </label>
                        <label>Name: <input type="text" name="member_name" placeholder="Name & Details" required></label>
                        <label>ID Number (Compulsory): <input type="text" name="national_id" placeholder="National ID" required></label>
                        <label>Phone Number (Optional): <input type="text" name="phone_number" placeholder="Phone Number"></label>
                        <button type="submit" name="add_member" style="height: 34px;">Add Member</button>
                    </form>
                    
                    <h4 style="margin-top: 20px;">Existing Members</h4>
                    <table class="member-table">
                        <thead>
                            <tr>
                                <th>M/NO</th><th>Group</th><th>Name</th><th>National ID</th><th>Phone No.</th><th>Active?</th><th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($all_members as $member): ?>
                                <tr>
                                    <td><?php echo $member['member_no']; ?></td>
                                    <form method="POST" action="admin.php" style="display: inline;">
                                        <input type="hidden" name="member_no" value="<?php echo $member['member_no']; ?>">
                                    <td>
                                        <select name="group_id" style="width: 100px;">
                                            <?php foreach ($groups_list as $group): ?>
                                                <option value="<?php echo $group['group_id']; ?>" <?php echo ($group['group_id'] == $member['group_id']) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($group['group_name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td><input type="text" name="member_name" value="<?php echo htmlspecialchars($member['name_details']); ?>" style="width: 150px;"></td>
                                    <td><input type="text" name="national_id" value="<?php echo htmlspecialchars($member['national_id']); ?>" required style="width: 100px;"></td>
                                    <td><input type="text" name="phone_number" value="<?php echo htmlspecialchars($member['phone_number'] ?? ''); ?>" style="width: 100px;"></td>
                                    <td>
                                        <input type="checkbox" name="is_active" value="1" <?php echo ($member['is_active'] == 1) ? 'checked' : ''; ?>>
                                    </td>
                                    <td>
                                        <button type="submit" name="update_member" class="edit-btn small-btn">Update</button>
                                    </form>
                                        <a href="admin.php?delete_member_no=<?php echo $member['member_no']; ?>" class="action-link delete-btn small-btn" 
                                           onclick="return confirm('WARNING: Deleting a member removes them entirely. Continue?');">Delete</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </details>
        </div>
    </div>
    
    <?php close_db_connection($conn); ?>
</body>
</html>