// scripts.js - Handles real-time calculations and tab switching for the RLF Ledger

document.addEventListener('DOMContentLoaded', () => {
    // 1. Initial Calculation on Load
    // Runs to initialize C/F columns and Grand Totals based on loaded B/F balances.
    calculateAll();
    
    // 2. Add event listener to the entire table for input changes
    const memberTable = document.getElementById('memberTable');
    if (memberTable) {
        // Event delegation: listen for input events on the table body
        memberTable.addEventListener('input', (event) => {
            // Only recalculate if the input field is a number input
            if (event.target.tagName === 'INPUT' && event.target.type === 'number') {
                calculateAll();
            }
        });
    }

    // 3. Add event listener to summary expense fields
    const summaryFields = document.querySelectorAll('#summary input[type="number"]');
    summaryFields.forEach(input => {
        input.addEventListener('input', calculateSummary);
    });

    // 4. Force calculation on initial load to set all grand totals correctly,
    // especially if B/F balances were loaded.
    const groupSelect = document.querySelector('select[name="group_id"]');
    if (groupSelect && groupSelect.value) {
        calculateAll(); 
    }
});

// --- Core Calculation Functions ---

/**
 * Parses input value to a float, defaulting to 0.00 if invalid or empty.
 * @param {string} value - The input value.
 * @returns {number} The parsed float number.
 */
function parseValue(value) {
    const num = parseFloat(value);
    return isNaN(num) ? 0.00 : num;
}

/**
 * Calculates all row totals and grand totals.
 * CRITICAL: This is the function that bridges Page 1 data to Page 2 totals.
 */
function calculateAll() {
    const tableBody = document.querySelector('#memberTable tbody');
    if (!tableBody) return;

    // Grand Totals initialization
    let totalRepaid = 0;
    let totalShares = 0;
    let totalFines = 0;
    let totalAdvancePaid = 0; // ADDED
    let totalWithdrawal = 0;
    let totalNewLoan = 0;
    let totalNewAdvance = 0;

    const rows = tableBody.querySelectorAll('tr');

    rows.forEach(row => {
        const rowData = calculateMember(row);

        // Accumulate Grand Totals
        totalRepaid += rowData.totalRepaid; // Principal + Interest + Advance Paid
        totalShares += rowData.shares;
        totalFines += rowData.fines;
        totalAdvancePaid += rowData.advancePaid; // ADDED
        totalWithdrawal += rowData.withdrawal;
        totalNewLoan += rowData.newLoan;
        totalNewAdvance += rowData.newAdvance;
    });

    // Update Summary Section Totals using the accumulated group totals
    updateSummarySection(
        totalRepaid, 
        totalShares, 
        totalFines, 
        totalAdvancePaid, // ADDED
        totalWithdrawal, 
        totalNewLoan, 
        totalNewAdvance
    );
}

/**
 * Calculates a single member's row data (C/F balances, Total Repaid).
 * @param {HTMLElement} rowElement - The table row element.
 * @returns {object} The row data including totals.
 */
function calculateMember(rowElement) {
    // Safely get the row element
    const row = rowElement.closest('tr') || rowElement;

    // Get input and output elements by class name within the row, using optional chaining for safety
    const bfSavings = parseValue(row.querySelector('.bf-savings')?.value);
    const bfLoan = parseValue(row.querySelector('.bf-loan')?.value);
    
    const principal = parseValue(row.querySelector('.principal')?.value);
    const interest = parseValue(row.querySelector('.loan-interest')?.value);
    const shares = parseValue(row.querySelector('.shares')?.value);
    const fines = parseValue(row.querySelector('.fines')?.value);
    const advancePaid = parseValue(row.querySelector('.advance-paid')?.value); // ADDED
    
    const withdrawal = parseValue(row.querySelector('.withdrawal')?.value);
    const newLoan = parseValue(row.querySelector('.new-loan')?.value);
    const newAdvance = parseValue(row.querySelector('.new-advance')?.value);
    
    // Total Repaid must now include Principal, Interest, AND Advance Paid
    const totalRepaid = principal + interest + advancePaid; // MODIFIED
    
    // C/F Calculations
    const cfSavings = bfSavings + shares - withdrawal;
    // Loan C/F = B/F Loan - Principal - Advance Paid + New Loan + New Advance 
    const cfLoan = bfLoan - principal - advancePaid + newLoan + newAdvance; // MODIFIED
    
    // Update the Read-Only/Span fields in the row
    if (row.querySelector('.total-repaid')) row.querySelector('.total-repaid').value = totalRepaid.toFixed(2);
    if (row.querySelector('.cf-savings')) row.querySelector('.cf-savings').textContent = cfSavings.toFixed(2);
    if (row.querySelector('.cf-loan')) row.querySelector('.cf-loan').textContent = cfLoan.toFixed(2);

    return { 
        totalRepaid, 
        shares, fines, 
        advancePaid, // ADDED
        withdrawal, newLoan, newAdvance 
    };
}

/**
 * Updates the totals displayed in the Group Summary tab (Page 2).
 */
function updateSummarySection(totalRepaid, totalShares, totalFines, totalAdvancePaid, totalWithdrawal, totalNewLoan, totalNewAdvance) { // ADDED totalAdvancePaid
    
    // Total Inflow = Total Repaid (which now includes Adv. Paid) + Shares + Fines
    const totalInflow = totalRepaid + totalShares + totalFines;
    const totalDisbursements = totalWithdrawal + totalNewLoan + totalNewAdvance;
    
    // 2. Update Inflow Totals (Page 2 elements)
    if (document.getElementById('summaryTotalPaid')) document.getElementById('summaryTotalPaid').textContent = totalInflow.toFixed(2);
    if (document.getElementById('summaryTotalRepaid')) document.getElementById('summaryTotalRepaid').textContent = (totalRepaid).toFixed(2); 
    if (document.getElementById('summaryTotalSavings')) document.getElementById('summaryTotalSavings').textContent = totalShares.toFixed(2);
    if (document.getElementById('summaryTotalFines')) document.getElementById('summaryTotalFines').textContent = totalFines.toFixed(2);
    if (document.getElementById('summaryTotalAdvancePaid')) document.getElementById('summaryTotalAdvancePaid').textContent = totalAdvancePaid.toFixed(2); // ADDED

    // 3. Update Outflow Totals (Page 2 elements)
    if (document.getElementById('totalWithdrawal')) document.getElementById('totalWithdrawal').textContent = totalWithdrawal.toFixed(2);
    if (document.getElementById('totalNewLoan')) document.getElementById('totalNewLoan').textContent = totalNewLoan.toFixed(2);
    if (document.getElementById('totalNewAdvance')) document.getElementById('totalNewAdvance').textContent = totalNewAdvance.toFixed(2);
    
    // 4. Calculate Final Cash Reconciliation
    calculateSummary();
}

/**
 * Calculates the final cash reconciliation (Grand Total) in the Summary tab.
 */
function calculateSummary() {
    // A. Calculated Totals (from member ledger, read from Page 2 spans)
    const totalInflow = parseValue(document.getElementById('summaryTotalPaid')?.textContent);
    const totalDisbursements = parseValue(document.getElementById('totalWithdrawal')?.textContent) + 
                               parseValue(document.getElementById('totalNewLoan')?.textContent) + 
                               parseValue(document.getElementById('totalNewAdvance')?.textContent);

    // B. Manual Inputs (Expenses + Previous Banking)
    const prvBanking = parseValue(document.getElementById('prvBanking')?.value);
    
    let totalExpenses = 0;
    document.querySelectorAll('.expense').forEach(input => {
        totalExpenses += parseValue(input.value);
    });

    // C. Final Reconciliation
    // Cash_Balance = Prev_Banking + Total_Inflow - Total_Disbursements - Total_Expenses
    const grandTotal = prvBanking + totalInflow - totalDisbursements - totalExpenses;

    // Update Display fields
    if (document.getElementById('totalExpenses')) document.getElementById('totalExpenses').textContent = totalExpenses.toFixed(2);
    if (document.getElementById('grandTotal')) document.getElementById('grandTotal').textContent = grandTotal.toFixed(2);
    if (document.getElementById('grandTotalSubmitted')) document.getElementById('grandTotalSubmitted').value = grandTotal.toFixed(2); 
}

// --- Tab Switching Function ---
function showTab(tabId) {
    // Hide all tab contents
    document.querySelectorAll('.tab-content').forEach(content => {
        content.classList.remove('active');
    });
    // Deactivate all buttons
    document.querySelectorAll('.tab-button').forEach(button => {
        button.classList.remove('active');
    });

    // Show the selected tab content and activate the button
    document.getElementById(tabId).classList.add('active');
    document.querySelector(`.tab-button[onclick="showTab('${tabId}')"]`).classList.add('active');
    
    // Recalculate summary every time the Summary tab is viewed to ensure fresh totals
    if (tabId === 'summary') {
        calculateAll(); // Forces calculation of all group totals from Page 1 data
        calculateSummary(); // Forces calculation of final cash balance
    }
}