<?php
// reports.php - Dedicated Report View
require_once 'db_model.php'; 

global $conn;

// --- 1. Get Filter Parameters ---
$group_id = isset($_GET['group_id']) && is_numeric($_GET['group_id']) ? intval($_GET['group_id']) : null;
$date_from = $_GET['date_from'] ?? null;
$date_to = $_GET['date_to'] ?? null;
$view_date = $_GET['view_date'] ?? null; // Specific date to view details for

// --- 2. Load Data ---
$groups_list = get_all_groups($conn);
$report_summaries = get_group_summaries_by_filters($conn, $group_id, $date_from, $date_to);
$detailed_transactions = [];
$selected_summary = null;

if ($view_date && $group_id) {
    // If a specific date is selected for detail view, fetch the transactions
    $detailed_transactions = get_detailed_member_transactions($conn, $view_date, $group_id);
    
    // Find the summary row corresponding to the detailed view date
    foreach ($report_summaries as $summary) {
        if (isset($summary['report_date']) && $summary['report_date'] === $view_date) {
            $selected_summary = $summary;
            break;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>RLF System - Monthly Reports</title>
    <link rel="stylesheet" href="styles.css">
    <style>
        .container { max-width: 1400px; margin: 40px auto; padding: 20px; background-color: #424242; border-radius: 8px; box-shadow: 0 0 10px rgba(0, 0, 0, 0.5); color: #fff; }
        .filter-bar { background-color: #555; padding: 15px; margin-bottom: 20px; border-radius: 5px; display: flex; flex-wrap: wrap; gap: 15px; align-items: flex-end; }
        .filter-bar label { color: #ffb74d; font-weight: bold; }
        .filter-bar input[type="date"], .filter-bar select { padding: 8px; border: 1px solid #7c43bd; border-radius: 4px; background-color: #333; color: #fff; }
        .report-table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        .report-table th, .report-table td { border: 1px solid #666; padding: 10px; text-align: left; background-color: #4e4e4e; }
        .report-table th { background-color: #4a148c; color: white; }
        .view-link { background-color: #ffb74d; color: black; padding: 5px 10px; border-radius: 4px; text-decoration: none; font-weight: bold; display: inline-block; }
        .detail-view { margin-top: 30px; padding: 20px; border: 2px solid #8bc34a; border-radius: 8px; background-color: #383838; }
        .summary-box { display: flex; flex-wrap: wrap; gap: 20px; margin-bottom: 20px; }
        .summary-item { background-color: #555; padding: 10px; border-radius: 5px; flex: 1 1 200px; }
        .summary-item strong { color: #8bc34a; display: block; font-size: 1.2em; }
    </style>
</head>
<body>
    <header>
        <h1>RLF System - Monthly Reports & History</h1>
    </header>

    <div class="container">
        <p style="text-align: center;">
            <a href="index.php" style="color: #ffb74d;">← Back to Ledger Entry</a> 
            <a href="admin.php" style="color: #ffb74d; margin-left: 20px;">Manage Groups & Members</a>
        </p>

        <div class="filter-bar">
            <form method="GET" action="reports.php" style="display: contents;">
                <label>Filter Group:
                    <select name="group_id">
                        <option value="">-- All Groups --</option>
                        <?php foreach ($groups_list as $group): ?>
                            <option value="<?php echo $group['group_id']; ?>" <?php echo ($group['group_id'] == $group_id) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($group['group_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </label>
                <label>From Date: <input type="date" name="date_from" value="<?php echo htmlspecialchars($date_from); ?>"></label>
                <label>To Date: <input type="date" name="date_to" value="<?php echo htmlspecialchars($date_to); ?>"></label>
                <button type="submit" style="background-color: #6a1b9a; color: white; padding: 8px 12px; border: none; border-radius: 4px; cursor: pointer;">Apply Filters</button>
            </form>
        </div>

        <h3>Available Monthly Reports (<?php echo count($report_summaries); ?> found)</h3>
        
        <?php if (empty($report_summaries)): ?>
            <p style="padding: 15px; background-color: #555; border-radius: 5px;">No reports found matching the criteria. Try adjusting the filters.</p>
        <?php else: ?>
            <table class="report-table">
                <thead>
                    <tr>
                        <th>Report Date</th>
                        <th>Group Name</th>
                        <th>Officer</th>
                        <th>Total Expenses</th>
                        <th>Ending Cash Balance</th>
                        <th>Next Meeting</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($report_summaries as $summary): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($summary['report_date']); ?></td>
                            <td><?php echo htmlspecialchars($summary['group_name']); ?></td>
                            <td><?php echo htmlspecialchars($summary['officer_name']); ?></td>
                            <td><?php echo number_format($summary['expense_meals'] + $summary['expense_transport'] + $summary['expense_stationary'] + $summary['expense_others'], 2); ?></td>
                            <td>**<?php echo number_format($summary['final_grand_total'], 2); ?>**</td>
                            <td><?php echo htmlspecialchars($summary['next_date']); ?> @ <?php echo htmlspecialchars($summary['next_venue']); ?></td>
                            <td>
                                <a href="?group_id=<?php echo $summary['group_id']; ?>&view_date=<?php echo $summary['report_date']; ?>" class="view-link">View Details</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>

        <?php if ($view_date && $selected_summary): ?>
            <div class="detail-view">
                <h3>Detailed Report for **<?php echo htmlspecialchars($selected_summary['group_name']); ?>** on **<?php echo htmlspecialchars($view_date); ?>**</h3>
                
                <div class="summary-box">
                    <div class="summary-item">Cash Balance B/F: <strong><?php echo number_format($selected_summary['prv_banking'], 2); ?></strong></div>
                    <div class="summary-item">Total Expenses: <strong><?php echo number_format($selected_summary['expense_meals'] + $selected_summary['expense_transport'] + $selected_summary['expense_stationary'] + $selected_summary['expense_others'], 2); ?></strong></div>
                    <div class="summary-item">Final Grand Total: <strong><?php echo number_format($selected_summary['final_grand_total'], 2); ?></strong></div>
                    <div class="summary-item">Next Stage: <strong><?php echo htmlspecialchars($selected_summary['next_stage']); ?></strong></div>
                </div>

                <h4>Member Transaction Details:</h4>
                <table class="report-table">
                    <thead>
                        <tr>
                            <th>M/NO</th>
                            <th>Name</th>
                            <th>National ID</th> 
                            <th>Phone No.</th> 
                            <th>B/F Savings</th>
                            <th>B/F Loan</th>
                            <th>Principal Paid</th>
                            <th>Shares</th>
                            <th>Advance Paid</th> <th>Withdrawal</th>
                            <th>New Loan</th>
                            <th>New Advance</th>
                            <th>C/F Savings</th>
                            <th>C/F Loan</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($detailed_transactions as $tx): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($tx['member_no']); ?></td>
                            <td><?php echo htmlspecialchars($tx['name_details']); ?></td>
                            <td><?php echo htmlspecialchars($tx['national_id']); ?></td> 
                            <td><?php echo htmlspecialchars($tx['phone_number'] ?? '-'); ?></td> 
                            <td><?php echo number_format($tx['bf_savings'], 2); ?></td>
                            <td><?php echo number_format($tx['bf_loan'], 2); ?></td>
                            <td><?php echo number_format($tx['principal_paid'], 2); ?></td>
                            <td><?php echo number_format($tx['shares_this_month'], 2); ?></td>
                            <td><?php echo number_format($tx['advance_paid_today'] ?? 0, 2); ?></td> <td><?php echo number_format($tx['savings_withdrawal'], 2); ?></td>
                            <td><?php echo number_format($tx['new_loan'], 2); ?></td>
                            <td><?php echo number_format($tx['new_advance'], 2); ?></td>
                            <td>**<?php echo number_format($tx['cf_savings'], 2); ?>**</td>
                            <td>**<?php echo number_format($tx['cf_loan'], 2); ?>**</td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php elseif ($view_date && !$selected_summary): ?>
             <p style="padding: 15px; background-color: #f44336; color: white; border-radius: 5px;">Error: Could not find the detailed group summary for the selected date.</p>
        <?php endif; ?>

    </div>
    
    <?php close_db_connection($conn); ?>
</body>
</html>