<?php
// ledger_controller.php
require_once 'db_model.php';

// --- Global Data Structures ---
global $conn;

$groups_list = get_all_groups($conn);
$selected_group_id = $_GET['group_id'] ?? ($_POST['group_id'] ?? null); 
$members_data = []; 
$report_status = ''; 

// --- Helper: Get Group Name for Display ---
function get_group_name_by_id($groups, $id) {
    foreach($groups as $group) {
        if ($group['group_id'] == $id) return $group['group_name'];
    }
    return '';
}

// --- 1. Load Initial Member Data (B/F Balances) ---
function load_ledger_data() {
    global $conn, $members_data, $selected_group_id;
    
    if (!$selected_group_id) {
        return; 
    }
    
    $members = get_all_active_members_by_group($conn, $selected_group_id);
    
    foreach ($members as $member) {
        $balances = get_last_report_balances($conn, $member['member_no']);
        
        $members_data[] = [
            'member_no' => $member['member_no'], 
            'name_details' => $member['name_details'],
            'bf_savings' => $balances['cf_savings'], 
            'bf_loan' => $balances['cf_loan']       
        ];
    }
}

// --- 2. Process Form Submission ---
function handle_submission() {
    global $conn, $report_status, $selected_group_id; 
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') return;
    
    $report_date = $_POST['reportDate'] ?? date('Y-m-d');
    
    $conn->begin_transaction();
    $transaction_successful = true;

    // A. Loop through all submitted member transactions
    if (isset($_POST['member_no'])) {
        foreach ($_POST['member_no'] as $key => $member_no) { 
            // Retrieve and sanitize data
            $member_no = intval($member_no); 
            $bf_savings = floatval($_POST['bf_savings'][$key]);
            $bf_loan = floatval($_POST['bf_loan'][$key]);
            $principal = floatval($_POST['principal'][$key]);
            $loan_interest = floatval($_POST['loan_interest'][$key]); 
            $shares = floatval($_POST['shares'][$key]);
            $fines = floatval($_POST['fines'][$key]);
            $withdrawal = floatval($_POST['withdrawal'][$key]);
            $new_loan = floatval($_POST['new_loan'][$key]);
            $new_advance = floatval($_POST['new_advance'][$key]);
            $advance_paid = floatval($_POST['advance_paid'][$key]); // ADDED
            
            // Server-side C/F Calculations 
            $cf_savings = $bf_savings + $shares - $withdrawal;
            // Advance payment acts like Principal paid, Advance issue acts like New Loan issued
            $cf_loan = $bf_loan - $principal - $advance_paid + $new_loan + $new_advance; // REVISED

            $member_data = [
                'member_no' => $member_no, 
                'report_date' => $report_date,
                'bf_savings' => $bf_savings,
                'bf_loan' => $bf_loan,
                'principal' => $principal,
                'loan_interest' => $loan_interest,
                'shares' => $shares,
                'fines' => $fines,
                'withdrawal' => $withdrawal,
                'new_loan' => $new_loan,
                'new_advance' => $new_advance,
                'advance_paid' => $advance_paid, // ADDED
                'cf_savings' => $cf_savings,
                'cf_loan' => $cf_loan,
            ];

            if (!save_member_transaction($conn, $member_data)) {
                $transaction_successful = false;
                break;
            }
        }
    }
    
    // B. Save Group Summary Data 
    if ($transaction_successful) {
        $summary_data = [
            'report_date' => $report_date,
            'groupName' => $_POST['groupName'] ?? 'N/A',
            'officerName' => $_POST['officerName'] ?? 'N/A',
            'prvBanking' => floatval($_POST['prvBanking'] ?? 0),
            'expenseMeals' => floatval($_POST['expenseMeals'] ?? 0),
            'expenseTransport' => floatval($_POST['expenseTransport'] ?? 0),
            'expenseStationary' => floatval($_POST['expenseStationary'] ?? 0),
            'expenseOthers' => floatval($_POST['expenseOthers'] ?? 0),
            'nextDate' => $_POST['nextDate'] ?? null,
            'nextTime' => $_POST['nextTime'] ?? null,
            'nextVenue' => $_POST['nextVenue'] ?? 'N/A',
            'nextStage' => $_POST['nextStage'] ?? 'N/A',
            'grandTotal' => floatval($_POST['grandTotalSubmitted'] ?? 0) // Saved from JS calculation
        ];

        if (!save_group_summary($conn, $summary_data)) {
            $transaction_successful = false;
        }
    }
    
    // C. Finalize Transaction
    if ($transaction_successful) {
        $conn->commit();
        $report_status = '<div style="color: green; font-weight: bold; background: #e8f5e9; padding: 10px;">SUCCESS: Monthly report saved successfully!</div>';
        // Redirect to clear POST data
        header("Location: index.php?group_id=" . $selected_group_id . "&status=success"); 
        exit;
    } else {
        $conn->rollback();
        $report_status = '<div style="color: red; font-weight: bold; background: #ffebee; padding: 10px;">ERROR: Failed to save the report. Transaction rolled back.</div>';
    }
}

// Execute logic
handle_submission();
load_ledger_data();
?>