<?php
// db_model.php
require_once 'config.php';

// --- Group CRUD Operations ---

function get_all_groups($conn) {
    $sql = "SELECT group_id, group_name FROM groups ORDER BY group_name ASC";
    $result = $conn->query($sql);
    $groups = [];
    if ($result && $result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $groups[] = $row;
        }
    }
    return $groups;
}

function add_new_group($conn, $group_name) {
    $sql = "INSERT INTO groups (group_name, creation_date) VALUES (?, CURDATE())";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("s", $group_name);
        return $stmt->execute();
    }
    return false;
}

function update_group($conn, $group_id, $group_name) {
    $sql = "UPDATE groups SET group_name = ? WHERE group_id = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("si", $group_name, $group_id);
        return $stmt->execute();
    }
    return false;
}

function delete_group($conn, $group_id) {
    $sql = "DELETE FROM groups WHERE group_id = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $group_id);
        return $stmt->execute();
    }
    return false;
}

// --- Member CRUD & Retrieval Operations (Uses member_no) ---

function get_all_members($conn) {
    $sql = "SELECT m.*, g.group_name FROM members m 
            LEFT JOIN groups g ON m.group_id = g.group_id 
            ORDER BY g.group_name, m.member_no ASC";
    $result = $conn->query($sql);
    $members = [];
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $members[] = $row;
        }
    }
    return $members;
}

function get_all_active_members_by_group($conn, $group_id) {
    $sql = "SELECT member_no, name_details FROM members 
            WHERE group_id = ? AND is_active = 1 
            ORDER BY member_no ASC";
            
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $group_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $members = [];
        if ($result->num_rows > 0) {
            while($row = $result->fetch_assoc()) {
                $members[] = $row;
            }
        }
        $stmt->close();
    }
    return $members;
}

function add_new_member($conn, $group_id, $name_details, $national_id, $phone_number) {
    $sql = "INSERT INTO members (group_id, name_details, national_id, phone_number, is_active) 
            VALUES (?, ?, ?, ?, TRUE)";

    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("isss", $group_id, $name_details, $national_id, $phone_number);
        return $stmt->execute();
    }
    return false;
}

function update_member($conn, $member_no, $group_id, $name_details, $national_id, $phone_number, $is_active) {
    $sql = "UPDATE members SET group_id = ?, name_details = ?, national_id = ?, phone_number = ?, is_active = ? WHERE member_no = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("isssii", $group_id, $name_details, $national_id, $phone_number, $is_active, $member_no);
        return $stmt->execute();
    }
    return false;
}

function delete_member($conn, $member_no) {
    $sql = "DELETE FROM members WHERE member_no = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $member_no);
        return $stmt->execute();
    }
    return false;
}

// --- Transaction Functions (Uses member_no) ---

function get_last_report_balances($conn, $member_no) {
    $sql = "SELECT cf_savings, cf_loan 
            FROM monthly_transactions 
            WHERE member_no = ? 
            ORDER BY report_date DESC, transaction_id DESC 
            LIMIT 1";
            
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $member_no);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            return $result->fetch_assoc();
        }
        $stmt->close();
    }
    return ['cf_savings' => 0.00, 'cf_loan' => 0.00];
}

function save_member_transaction($conn, $data) {
    $sql = "INSERT INTO monthly_transactions (
        member_no, report_date, bf_savings, bf_loan, 
        principal_paid, loan_interest, shares_this_month, fines_charges, 
        savings_withdrawal, new_loan, new_advance, advance_paid_today,
        cf_savings, cf_loan
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

    if ($stmt = $conn->prepare($sql)) {
        // 'i' for member_no (int), 's' for date (string), 'd' for decimals (12 decimals)
        $stmt->bind_param("isddddddddddd", 
            $data['member_no'], $data['report_date'], $data['bf_savings'], $data['bf_loan'], 
            $data['principal'], $data['loan_interest'], $data['shares'], $data['fines'], 
            $data['withdrawal'], $data['new_loan'], $data['new_advance'], $data['advance_paid'], // Mapped 'advance_paid' from controller to 'advance_paid_today' in DB
            $data['cf_savings'], $data['cf_loan']
        );
        return $stmt->execute();
    }
    return false;
}

function save_group_summary($conn, $data) {
    $sql = "INSERT INTO group_summary (
        report_date, group_name, officer_name, prv_banking, 
        expense_meals, expense_transport, expense_stationary, expense_others, 
        next_date, next_time, next_venue, next_stage, final_grand_total
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    if ($stmt = $conn->prepare($sql)) {
        // 's' for string/date, 'd' for decimals
        $stmt->bind_param("ssddddddssssd", 
            $data['report_date'], $data['groupName'], $data['officerName'], $data['prvBanking'], 
            $data['expenseMeals'], $data['expenseTransport'], $data['expenseStationary'], $data['expenseOthers'], 
            $data['nextDate'], $data['nextTime'], $data['nextVenue'], $data['nextStage'], $data['grandTotal']
        );
        return $stmt->execute();
    }
    return false;
}

// --- Report Retrieval Functions (FOR reports.php) ---

function get_group_summaries_by_filters($conn, $group_id = null, $date_from = null, $date_to = null) {
    $sql = "SELECT gs.*, g.group_id, g.group_name 
            FROM group_summary gs
            JOIN groups g ON gs.group_name = g.group_name
            WHERE 1=1";
    $params = [];
    $types = "";

    if ($group_id) {
        $sql .= " AND g.group_id = ?";
        $params[] = $group_id;
        $types .= "i";
    }
    if ($date_from && $date_to) {
        $sql .= " AND gs.report_date BETWEEN ? AND ?";
        $params[] = $date_from;
        $params[] = $date_to;
        $types .= "ss";
    } else {
        // Default to showing reports from the last 6 months if no date range is set
        $sql .= " AND gs.report_date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)";
    }
    
    $sql .= " ORDER BY gs.report_date DESC";
    
    $stmt = $conn->prepare($sql);
    if ($stmt) {
        if (!empty($params)) {
            $stmt->bind_param($types, ...$params);
        }
        $stmt->execute();
        $result = $stmt->get_result();
        $summaries = [];
        if ($result) {
            while ($row = $result->fetch_assoc()) {
                $summaries[] = $row;
            }
        }
        $stmt->close();
        return $summaries;
    }
    return [];
}

function get_detailed_member_transactions($conn, $report_date, $group_id) {
    // This query fetches transactions AND the associated member's details
    $sql = "SELECT mt.*, m.member_no, m.name_details, m.national_id, m.phone_number 
            FROM monthly_transactions mt
            JOIN members m ON mt.member_no = m.member_no
            WHERE mt.report_date = ? 
            AND m.group_id = ?
            ORDER BY m.member_no ASC";
            
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("si", $report_date, $group_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $transactions = [];
        if ($result->num_rows > 0) {
            while($row = $result->fetch_assoc()) {
                $transactions[] = $row;
            }
        }
        $stmt->close();
        return $transactions;
    }
    return [];
}
?>